/* -*-C-*-
 ##############################################################################
 #
 # File:        demo/rpmtrig2.c
 # RCS:         $Id: rpmtrig2.c,v 1.24 1998/07/20 22:59:30 ericb Exp $
 # Description: Host program for doing RPM arm/trigger across multiple E1432s
 # Created:     October 20, 1995
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This is a host program that synchronizes RPM triggering across two E1432s.
 # It will display the time waveforms of the first 4 channels of each E1432.
 # It will print out the RPM values of both tach channels at the trigger 
 # points.
 #
 # The equipment required for this demo is:
 # 	1. VXI card cage with MXI interface to host
 #  	2. one E1432/33 with a tachometer option AYD
 #	3. another E1432/33 
 #
 # Set the logical address of the the E1432 with tachometer to the address
 # specified in the LA1 defined below (default 8) and set the other E1432 to
 # the address specified in the LA2 defined below (default 9).
 #
 # Hook up a sweeping signal generator as such:
 #    output set to 2 volts peak sine wave.
 #    tee output to go to channel 1 inputs of both modules as well and into
 #    both tachometer inputs on the second module.
 #
 # To run the measurement:
 #    start the program
 #    sweep the frequency between 5Hz(300 RPM) and 105Hz (6300 RPM)
 #    over 30 seconds
 #
 # The program should display the waveform in the channel 1 and 5 displays on
 # the screen.  The measurement pre-arms at 500 RPM and begins triggering and
 # updating the display at 700 RPM and every 100 RPM interval after that 
 # until 6000 rpm.  If the trigger delay and trigger mode are left at the
 # defaults ( -BLOCK_SIZE/2 and E1432_TACH_EDGE_TRIGGER) the center point of
 # the block of data will remain stationary since we are triggering on a
 # tach edge and taking data +/- BLOCK_SIZE/2 on either side of that trigger
 # point.  In addition, the arming RPM points, RPM at trigger point of arming
 # tach channel, and average RPM at during the data block of other tach 
 # channel are displayed on the terminal window.
 #
 # Note that the RPMs for the two channels are not exactly the same.  The
 # RPM of the arm/trigger channel is calculated using a quadratic fitting 
 # method across tach edge times near the trigger point, while the second
 # channel's RPM is calculated by taking the average RPM for the block.
 #
 # Altering some of the default measurement parameters below will change the
 # nature of the display.  In particular, changing the trigger mode to
 # E1432_AUTO_TRIGGER will produce a display whose center is the exact point
 # at which the RPM reached the arming RPM and won't be lined up with a zero
 # crossing of the tach.  
 #
 # Revisions:
 #
 #
 ##############################################################################
 */
#include <stdio.h>
#include <stdlib.h>
#include "e1432.h"
#include "xplot.h"

/*#define EXT_TRIG*/

#define WARNING_MAX	100

/* defines for size of XPLOT windows */
#define WIDTH           140
#define HEIGHT          80

/* default module addresses */
#define NUM_MODS	2
#define LA1		8
#define LA2		9
#define NUM_CHANS_1	8
#define NUM_CHANS_2	8

/* default measurement parameters */
#define BLOCK_SIZE	1024
#define SPAN		20000.0
#define RANGE		2.0
#define RPM_MIN		120.0
#define RPM_PRE_ARM	500.0
#define RPM_LOW		700.0
#define RPM_HIGH	6000.0
#define RPM_INTERVAL	25.0

#if 0
/* trigger at the next tach edge after arm rpm reached */
#define TRIGGER_MODE	E1432_TACH_EDGE_TRIGGER
#else
/* trigger at the next tach edge after arm rpm reached */
#define TRIGGER_MODE	E1432_AUTO_TRIGGER
#endif

/* trigger in the middle of the block */
#define TRIGGER_DELAY	-BLOCK_SIZE/2

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

int
main(void)
{
    int  i, j, chanStart2;
    LONGSIZ32 count;
    LONGSIZ32 num = 0;
    SHORTSIZ16 la[NUM_MODS] = {LA1, LA2};
    SHORTSIZ16 status, input_chans1, input_chans2, total_chans, nchan1, nchan2;
    SHORTSIZ16 num_tachs;
    SHORTSIZ16 total_list[34];
    SHORTSIZ16 tach_list[2*E1432_TACH_CHANS];
    SHORTSIZ16 all, inputs, tachs, tach0, tach1, arm, nonarm;
    E1432ID hw;
    float rpm;
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    FLOATSIZ32 *data[32];
    long points = BLOCK_SIZE;
    char *plotid[32];
    int row, col;  
    char geometry[80];
    char title[80];
    float temp, floatRange = RANGE;
    char *semabinloc = "/opt/e1432/lib/sema.bin";
    struct e1432_trailer trailer;
    SHORTSIZ16 meas_state;
    SHORTSIZ16 warning[WARNING_MAX];
    unsigned long warningCount;
    int module, checkWarnings;

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    for(i=0; i < NUM_MODS; i++)
    {
        (void) printf("Checking for E1432 with firmware at logical address %d ... ", 
							(int)la[i]);
        (void) fflush(stdout);
        error = e1432_get_hwconfig(1, &(la[i]), &cf);
        if (error)
        {
	    (void) printf("Not found.\n");
	    (void) printf("Installing firmware from %s into E1432 at la %d ... ",
					       semabinloc, (int)la[i]);
	    (void) fflush(stdout);
	    error = e1432_install(1, &(la[i]), 0, semabinloc);
	    if (error)
	    {
	        (void) printf("\ne1432_install failed and returned error %s\n");
	        exit(0);
	    }
	    (void) printf("Done.\n");
        }
        else
        {
	    (void) printf("Found.\n");
        }
    }
    (void) printf("\n");
    
    CHECK(e1432_print_errors(1));

    /* assign channel numbers */
    CHECK(e1432_assign_channel_numbers(NUM_MODS, la, &hw));

    /* find channel number of first input on second module */
    error = e1432_get_hwconfig(1, &la[0], &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(1);
    }
    input_chans1 = cf.input_chans;
    num_tachs = cf.tach_chans;

    nchan1 = (cf.input_chans < NUM_CHANS_1) ? cf.input_chans : NUM_CHANS_1;

    error = e1432_get_hwconfig(1, &la[1], &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }

    input_chans2 = cf.input_chans;
    num_tachs += cf.tach_chans;

    nchan2 = (cf.input_chans < NUM_CHANS_2) ? cf.input_chans : NUM_CHANS_2;

    total_chans = nchan1 + nchan2;
    printf("Using %d channels from module 1 and %d channels from module 2\n",
							nchan1, nchan2);

    /* Create channel group */
    for(i=0; i < nchan1; i++)
	total_list[i] = E1432_INPUT_CHAN(i+1);
    for(; i < total_chans; i++)
	total_list[i] = E1432_INPUT_CHAN(i+1+(input_chans1 - nchan1));
    for(; i < total_chans + num_tachs; i++)
	total_list[i] = E1432_TACH_CHAN(i - total_chans + 1);
    
    inputs = e1432_create_channel_group(hw, total_chans, total_list);
    if (inputs >= 0)
    {
	(void)printf("e1432_create_channel_group inputs returned %d\n", inputs);
	exit(0);
    }

    all = e1432_create_channel_group(hw, total_chans + num_tachs, total_list);
    if (all >= 0)
    {
	(void) printf("e1432_create_channel_group all returned %d\n", all);
	exit(0);
    }
    
    /* Initialize hardware things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_SOURCE_BNC,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, floatRange));


    /* initialize buffers and XPLOT stuff */
    row = col = 0;
    temp = (float)(points - 1);

    for(i=0; i < total_chans; i++) {
        data[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!data[i]) {
	  (void) printf("Couldn't malloc data array of %d points\n", points);
          exit(1);
        }

	/* initialize so XPLOT does throwup */
	for(j=0; j < points; j++) {
	   data[i][j] = 0.0;
	}
        /* standard HPUX kernal won't support enough xplot windows */
	if(i < 48)
	{
            (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
				(WIDTH + 20) * col, (HEIGHT + 40) * row ); 
	    (void) sprintf(title, "Channel %d", i + 1);

            plotid[i] = xplot_init_plot(data[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
            col++;
            if((col % 8) == 0) {
                col = 0;
                row++;
            }

            xplot_change_yautoscale(plotid[i], 0);
            xplot_set_xscale(plotid[i], (float)0.0, temp);
            xplot_set_yscale(plotid[i], floatRange, -floatRange);
            xplot_change_xlabel(plotid[i], "Samples");
            xplot_change_ylabel(plotid[i], "Volts");
            xplot_repaint(plotid[i]); 
	}
    }

    /* assign tach channels */
    for(i = 0; i < num_tachs; i++)
        tach_list[i] = E1432_TACH_CHAN(i+1);
    
    tach0 = tach_list[0];
    tach1 = tach_list[1];
    tachs = e1432_create_channel_group(hw, num_tachs, tach_list);
    if (tachs >= 0)
    {
	(void) printf("create_channel_group for tachs returned %d\n", tachs);
	exit(1);
    }

    arm = tach0;
    nonarm = tach1;

#if 0
    e1432_set_internal_debug(hw, all, 0x400);
#endif

    CHECK(e1432_set_span(hw, all, SPAN));
    CHECK(e1432_set_data_mode(hw, all, E1432_DATA_MODE_OVERLAP_BLOCK));

    /* set up all arm trigger stuff */
#ifdef EXT_TRIG
    CHECK(e1432_set_trigger_channel(hw, arm, E1432_CHANNEL_OFF));
    CHECK(e1432_set_trigger_channel(hw, nonarm, E1432_CHANNEL_ON));
#else
    CHECK(e1432_set_trigger_channel(hw, arm, E1432_CHANNEL_ON));
    CHECK(e1432_set_trigger_channel(hw, nonarm, E1432_CHANNEL_OFF));
#endif

    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_LOWER, 0.0));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_UPPER, 0.0));
    CHECK(e1432_set_pre_arm_mode(hw, arm, E1432_ARM_RPM_RUNUP)); 
    CHECK(e1432_set_arm_channel(hw, arm, E1432_CHANNEL_ON)); 
    CHECK(e1432_set_pre_arm_rpm(hw, arm, RPM_PRE_ARM)); 
    CHECK(e1432_set_rpm_low(hw, arm, RPM_LOW)); 
    CHECK(e1432_set_rpm_high(hw, arm, RPM_HIGH)); 
    CHECK(e1432_set_rpm_interval(hw, arm, RPM_INTERVAL)); 
    CHECK(e1432_set_arm_mode(hw, all, E1432_ARM_RPM_RUNUP)); 
    CHECK(e1432_set_tach_max_time(hw, tachs, 60.0 / RPM_MIN)); 

    /* set trigger master to first module */
    CHECK(e1432_set_trigger_master(hw, tach0, E1432_TRIGGER_MASTER_ON));

    /* set block start relative to trigger point */
    CHECK(e1432_set_trigger_delay(hw, all, TRIGGER_DELAY));

    /* trigger mode */
    CHECK(e1432_set_auto_trigger(hw, all, TRIGGER_MODE)); 


    /* stop any previous measurement */
    CHECK(e1432_finish_measure(hw, all)); 

    (void) printf("Continuous measurement, press Cntl C to terminate\n\n");

    /* Start measurement */
    CHECK(e1432_init_measure(hw, all));

    (void)sleep(1);

    /* print the initial rpm */
    (void)printf("current rpm:	");
    for(i=0; i < num_tachs; i++)
    {
        CHECK(e1432_get_current_rpm(hw, tach_list[i], &rpm));
    	(void) printf("%g		", rpm);
    }
    (void)printf("\n");

    (void) printf("Waiting for pre-arm RPM of %g ... ", RPM_PRE_ARM);
    (void) fflush(stdout);

    for(meas_state = 0; meas_state < E1432_MEAS_STATE_IDLE; )
    {
        CHECK(e1432_get_meas_state(hw, arm, &meas_state));
	status = e1432_block_available(hw, inputs);
	if(status) break;	/* go on if data is available */
    }
    (void) printf("PRE-ARMED.\n\n");

#if 0
    (void) printf("Waiting for initial arming RPM of %g ... ", RPM_LOW);
    (void) fflush(stdout);

    for(meas_state = 0; meas_state < E1432_MEAS_STATE_TRIGGER; )
    {
        CHECK(e1432_get_meas_state(hw, arm, &meas_state));
	status = e1432_block_available(hw, all);
	if(status) break;	/* go on if data is available */
    }
    (void) printf("ARMED.\n\n");
#endif

    (void) printf("TRIG RPM	TACH2 RPM\n");

    /* main measurement loop */
    while(1) {
        /* Wait for block available , handling triggers */
	do
	{
	    CHECK(e1432_get_meas_state(hw, all, &meas_state));
	    if(meas_state == E1432_MEAS_STATE_TESTED)
	    {
		(void) printf("Measurement finished.\n");
		exit(0);
	    }
	    
	    if(nchan2 > 0)
	    {
	        CHECK(e1432_send_tachs(hw, all, tach1, NULL, 0, &count));
	        CHECK(e1432_send_tachs(hw, all, tach0, NULL, 0, &count));
	    }

            CHECK(e1432_read_register(hw, tach0, 	
					E1432_IRQ_STATUS2_REG, &status));
	
	    /* send trigger to slave */
	    if(status & E1432_IRQ_TRIGGER)
	        CHECK(e1432_send_trigger(hw, all));

	    if(status & E1432_IRQ_MEAS_ERROR)
	    {
		if(status & E1432_STATUS2_TACH_OVERFLOW)
		    (void) printf("Tach buffer overflowed in module 1\n");
		else
		    (void) printf("Fifo overflowed in module 1\n");
		exit(1);
	    }

	    /* check for warnings in second module */
	    module = 1;
	    if((checkWarnings = status & E1432_IRQ_MEAS_WARNING) == 0)
	    {
	        if(nchan2)
	        {
	            /* check for warnings in second module */
	            module = 2;
	            CHECK(e1432_read_register(hw, total_list[nchan1], 	
				    E1432_IRQ_STATUS2_REG, &status));
	            if(status & E1432_IRQ_MEAS_ERROR)
	            {
			if(status & E1432_STATUS2_TACH_OVERFLOW)
		    	    (void) printf("Tach buffer overflowed in module 2\n");
			else
		    	    (void) printf("Fifo overflowed in module 2\n");
	                exit(1);
	            }
	        }

	        checkWarnings = status & E1432_IRQ_MEAS_WARNING;
	    }

	    if(checkWarnings)
	    {
		/* read out all measurement warnings */
		do
		{
		    CHECK(e1432_get_meas_warning(hw, all, 
						 warning, WARNING_MAX,
						 &warningCount));
	   
		    if(warningCount)
		    {
	       		(void) printf("Module %d: %d Warning", 
							module, warningCount);
			if(warningCount > 1) 
			    (void) printf("s");
			(void) printf(":\n");
		    }

           	    for(i=0; i < warningCount; i++)
	   	    {
			(void) printf("    %s\n",
				      e1432_get_warning_string(warning[i]));
	   	    }

		} while (warningCount > 0);
	    }

	}while(e1432_block_available(hw, all)  == 0); 

        if (status < 0) {
	    (void) printf("status error %d\n", status);
	    exit(0);
        }

	for(j=0; j < num_tachs; j++)
	{
	    CHECK(e1432_get_data_rpm(hw, tach_list[j], &rpm));
	    (void) printf("%-6.6g		", rpm);
	}
	(void) printf("\n", rpm);
        
	for(i=0; i < total_chans; i++) { /* read time data */
            error = e1432_read_float32_data(hw, total_list[i],
		    E1432_TIME_DATA, data[i], points, &trailer, &count);
            if(error) {
   	        (void) printf("e1432_read_float32_data %d failed with error %d\n",
								    i, error);
	        exit(1);
            }

	    if(i < 48)
	    {
                xplot_check_events(plotid[i]);
                xplot_data_update(plotid[i]); 
	    }
        }

	if((++num % 20) == 0)
    	    (void) printf("\nTRIG RPM	TACH2 RPM\n");
    }

}
